import { jsx, jsxs } from 'react/jsx-runtime';
import React__default, { useRef, useEffect, useState, Children, cloneElement } from 'react';
import { CarretDown } from '@strapi/icons';
import { useCallbackRef } from '@strapi/ui-primitives';
import { NavLink } from 'react-router-dom';
import styled from 'styled-components';
import { getOptionStyle } from './utils.mjs';
import { KeyboardKeys } from '../helpers/keyboardKeys.mjs';
import { useId } from '../hooks/useId.mjs';
import { Link } from '../Link/Link.mjs';
import { Button } from '../Button/Button.mjs';
import { Popover } from '../Popover/Popover.mjs';
import { Box } from '../Box/Box.mjs';
import { Typography } from '../Typography/Typography.mjs';
import { Flex } from '../Flex/Flex.mjs';

const OptionButton = styled.button `
  border: none;
  padding: 0;
  background: transparent;
  cursor: pointer;
  ${getOptionStyle}
`;
const OptionLink = styled(NavLink) `
  text-decoration: none;
  ${getOptionStyle}
`;
const OptionExternalLink = styled(Link) `
  /* Removing Link hover effect */
  &:hover {
    color: currentColor;
  }

  &:focus-visible {
    /* Removing Link focus-visible after properties and reset to global outline */
    outline: 2px solid ${({ theme }) => theme.colors.primary600};
    outline-offset: 2px;
    &:after {
      content: none;
    }
  }

  ${getOptionStyle}
`;
const IconWrapper = styled.span `
  display: flex;
  align-items: center;
  svg {
    height: 4px;
    width: 6px;
  }
`;
const StyledButtonSmall = styled(Button) `
  padding: ${({ theme }) => `${theme.spaces[1]} ${theme.spaces[3]}`};
`;
const MenuItem = ({ children, onClick = () => { }, to, isFocused = false, href, ...props }) => {
    const menuItemRef = useRef();
    useEffect(() => {
        if (isFocused && menuItemRef.current) {
            menuItemRef.current.focus();
        }
    }, [isFocused]);
    const menuItemProps = {
        tabIndex: isFocused ? 0 : -1,
        ref: menuItemRef,
        role: 'menuitem',
        ...props,
    };
    const handleKeyDown = (e) => {
        if (e.key === KeyboardKeys.SPACE || e.key === KeyboardKeys.ENTER) {
            onClick();
        }
    };
    if (to && !href) {
        return (jsx(OptionLink, { to: to, ...menuItemProps, children: jsx(Box, { padding: 2, children: jsx(Typography, { children: children }) }) }));
    }
    if (href && !to) {
        return (jsx(OptionExternalLink, { isExternal: true, href: href, ...menuItemProps, children: jsx(Box, { padding: 2, children: jsx(Typography, { children: children }) }) }));
    }
    return (jsx(OptionButton, { onKeyDown: handleKeyDown, onMouseDown: onClick, type: "button", ...menuItemProps, children: jsx(Box, { padding: 2, children: jsx(Typography, { children: children }) }) }));
};
const SimpleMenu = ({ label, children, id, as: asComp, onOpen = () => { }, onClose = () => { }, size = 'M', popoverPlacement = 'bottom-start', onReachEnd, ...props }) => {
    const menuButtonRef = useRef(undefined);
    const menuId = useId(id);
    const didMount = useRef(false);
    const [visible, setVisible] = useState(false);
    const [focusedItemIndex, setFocusItem] = useState(0);
    const childrenArray = Children.toArray(children);
    const DefaultComponent = size === 'S' ? StyledButtonSmall : Button;
    const Component = asComp || DefaultComponent;
    const shouldHandleReachEnd = !!onReachEnd && typeof onReachEnd === 'function';
    useEffect(() => {
        if (['string', 'number'].includes(typeof label)) {
            // Useful to focus the selected item in the list
            const defaultItemIndexToFocus = childrenArray.findIndex((c) => (React__default.isValidElement(c) && c.props.children === label) || c === label);
            if (defaultItemIndexToFocus !== -1) {
                setFocusItem(defaultItemIndexToFocus);
            }
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [label]);
    const handleOpen = useCallbackRef(onOpen);
    const handleClose = useCallbackRef(onClose);
    useEffect(() => {
        if (didMount?.current) {
            if (visible) {
                handleOpen();
            }
            else {
                handleClose();
            }
        }
        else {
            didMount.current = true;
        }
    }, [didMount, handleClose, handleOpen, visible]);
    /* in case `label` is a custom react component, we know it is going to be
        a child of the menu button.
    */
    useEffect(() => {
        if (React__default.isValidElement(label) && focusedItemIndex === -1) {
            menuButtonRef.current.focus();
        }
    }, [label, focusedItemIndex]);
    const handleWrapperKeyDown = (e) => {
        if (visible) {
            if (e.key === KeyboardKeys.ESCAPE) {
                e.stopPropagation();
                setVisible(false);
                menuButtonRef.current.focus();
            }
            if (e.key === KeyboardKeys.DOWN) {
                setFocusItem((prev) => (prev === childrenArray.length - 1 ? 0 : prev + 1));
            }
            if (e.key === KeyboardKeys.UP) {
                setFocusItem((prev) => (prev === 0 ? childrenArray.length - 1 : prev - 1));
            }
        }
    };
    const handleKeyDown = (e) => {
        if (e.key === KeyboardKeys.ENTER || e.key === KeyboardKeys.SPACE) {
            setVisible((prevVisible) => !prevVisible);
        }
    };
    const handleBlur = (e) => {
        e.preventDefault();
        if (!e.currentTarget.contains(e.relatedTarget)) {
            setVisible(false);
        }
    };
    const handleMenuButtonMouseDown = (e) => {
        e.preventDefault();
        setVisible((prevVisible) => !prevVisible);
    };
    const handleReachEnd = () => {
        if (shouldHandleReachEnd) {
            onReachEnd();
        }
    };
    const childrenClone = childrenArray.map((child, index) => (
    // eslint-disable-next-line react/no-array-index-key
    jsx(Flex, { as: "li", justifyContent: "center", role: "menuitem", children: React__default.isValidElement(child)
            ? cloneElement(child, {
                onClick() {
                    child.props.onClick();
                    setVisible(false);
                    menuButtonRef.current.focus();
                },
                isFocused: focusedItemIndex === index,
            })
            : child }, index)));
    return (
    // TODO: review why we need to eslint it and how to solve this issue.
    // eslint-disable-next-line jsx-a11y/no-static-element-interactions
    jsxs("div", { onKeyDown: handleWrapperKeyDown, children: [jsx(Component, { label: React__default.isValidElement(label) ? null : label, "aria-haspopup": true, "aria-expanded": visible, "aria-controls": menuId, onKeyDown: handleKeyDown, onMouseDown: handleMenuButtonMouseDown, ref: menuButtonRef, type: "button", variant: "ghost", endIcon: jsx(IconWrapper, { children: jsx(CarretDown, { "aria-hidden": true }) }), ...props, children: label }), visible && (jsx(Popover, { onBlur: handleBlur, placement: popoverPlacement, source: menuButtonRef, onReachEnd: handleReachEnd, intersectionId: shouldHandleReachEnd ? `popover-${menuId}` : undefined, spacing: 4, children: jsx(Box, { role: "menu", as: "ul", padding: 1, id: menuId, children: childrenClone }) }))] }));
};

export { MenuItem, SimpleMenu };
