import { jsx, jsxs } from 'react/jsx-runtime';
import * as React from 'react';
import { Cross, CarretDown } from '@strapi/icons';
import { Select } from '@strapi/ui-primitives';
import styled, { css } from 'styled-components';
import { useComposedRefs } from '../hooks/useComposeRefs.mjs';
import { Flex } from '../Flex/Flex.mjs';
import { Box } from '../Box/Box.mjs';
import { getThemeSize, inputFocusStyle } from '../themes/utils.mjs';
import { Typography } from '../Typography/Typography.mjs';

const SelectTrigger = React.forwardRef(({ onClear, clearLabel = 'Clear', startIcon, disabled, hasError, size = 'M', children, ...restProps }, ref) => {
    const triggerRef = React.useRef(null);
    const handleClearClick = (e) => {
        if (onClear && !disabled) {
            onClear(e);
            triggerRef.current.focus();
        }
    };
    const composedRefs = useComposedRefs(triggerRef, ref);
    return (jsx(Select.Trigger, { asChild: true, children: jsxs(StyledTrigger, { "aria-disabled": disabled, "$hasError": hasError, "$size": size, ref: composedRefs, alignItems: "center", justifyContent: "space-between", position: "relative", overflow: "hidden", hasRadius: true, background: disabled ? 'neutral150' : 'neutral0', paddingLeft: 3, paddingRight: 3, gap: 4, cursor: "default", width: "100%", ...restProps, children: [jsxs(Flex, { flex: "1", as: "span", gap: 3, children: [startIcon && (jsx(Box, { as: "span", "aria-hidden": true, children: startIcon })), children] }), jsxs(Flex, { as: "span", gap: 3, children: [onClear ? (jsx(IconBox, { as: "button", hasRadius: true, background: "transparent", role: "button", tabIndex: 0, onClick: handleClearClick, "aria-disabled": disabled, "aria-label": clearLabel, title: clearLabel, cursor: "pointer", children: jsx(Cross, {}) })) : null, jsx(DownIcon, { children: jsx(CarretDown, {}) })] })] }) }));
});
const IconBox = styled(Box) `
  border: none;

  svg {
    height: ${11 / 16}rem;
    width: ${11 / 16}rem;
  }

  svg path {
    fill: ${({ theme }) => theme.colors.neutral600};
  }
`;
const StyledTrigger = styled(Flex) `
  border: 1px solid ${({ theme, $hasError }) => ($hasError ? theme.colors.danger600 : theme.colors.neutral200)};
  min-height: ${({ theme, $size }) => getThemeSize('input')({ theme, size: $size })};

  &[aria-disabled='true'] {
    color: ${(props) => props.theme.colors.neutral600};
  }

  /* Required to ensure the below inputFocusStyles are adhered too */
  &:focus-visible {
    outline: none;
  }

  ${({ theme, $hasError }) => inputFocusStyle()({ theme, hasError: $hasError })};
`;
const DownIcon = styled(Select.Icon) `
  & > svg {
    width: ${6 / 16}rem;

    & > path {
      fill: ${({ theme }) => theme.colors.neutral600};
    }
  }
`;
const SelectValue = React.forwardRef(({ children, placeholder, ...restProps }, ref) => (jsx(ValueType, { ref: ref, ellipsis: true, ...restProps, children: jsx(StyledValue, { placeholder: placeholder, children: children }) })));
const ValueType = styled(Typography) `
  flex: 1;
`;
const StyledValue = styled(Select.Value) `
  display: flex;
  gap: ${({ theme }) => theme.spaces[1]};
  flex-wrap: wrap;
`;
/* -------------------------------------------------------------------------------------------------
 * SelectContent
 * -----------------------------------------------------------------------------------------------*/
const SelectContent = styled(Select.Content) `
  background: ${({ theme }) => theme.colors.neutral0};
  box-shadow: ${({ theme }) => theme.shadows.filterShadow};
  border: 1px solid ${({ theme }) => theme.colors.neutral150};
  border-radius: ${({ theme }) => theme.borderRadius};
  min-width: var(--radix-select-trigger-width);
  /* This is from the design-system figma file. */
  max-height: 15rem;
  z-index: ${({ theme }) => theme.zIndices[1]};
`;
/* -------------------------------------------------------------------------------------------------
 * SelectViewport
 * -----------------------------------------------------------------------------------------------*/
const SelectViewport = styled(Select.Viewport) `
  padding: ${({ theme }) => theme.spaces[1]};
`;
const SelectItem = React.forwardRef((props, ref) => (jsx(StyledSelectItem, { ref: ref, ...props })));
const itemStyles = css `
  width: 100%;
  border: none;
  text-align: left;
  outline-offset: -3px;
  border-radius: ${(props) => props.theme.borderRadius};
  padding: ${(props) => `${props.theme.spaces[2]} ${props.theme.spaces[4]}`};
  padding-left: ${({ theme }) => theme.spaces[4]};
  background-color: ${({ theme }) => theme.colors.neutral0};
  display: flex;
  align-items: center;
  gap: ${({ theme }) => theme.spaces[2]};
  white-space: nowrap;
  user-select: none;

  &:focus-visible {
    outline: none;
    background-color: ${({ theme }) => theme.colors.primary100};
  }
`;
const StyledSelectItem = styled(Select.Item) `
  ${itemStyles}

  &:hover {
    background-color: ${({ theme }) => theme.colors.primary100};
  }

  &[data-state='checked'] {
    ${Typography} {
      font-weight: bold;
      color: ${({ theme }) => theme.colors.primary600};
    }
  }
`;
const Root = Select.Root;
const Trigger = SelectTrigger;
const Value = SelectValue;
const Portal = Select.Portal;
const Content = SelectContent;
const Viewport = SelectViewport;
const Item = SelectItem;
const ItemIndicator = Select.ItemIndicator;
const ItemText = Select.ItemText;
const Group = Select.Group;

export { Content, Group, Item, ItemIndicator, ItemText, Portal, Root, Trigger, Value, Viewport };
