import { jsx } from 'react/jsx-runtime';
import { useRef, useEffect } from 'react';
import { getFocusableNodes } from '../helpers/getFocusableNodes.mjs';
import { KeyboardKeys } from '../helpers/keyboardKeys.mjs';

const FocusTrap = ({ onEscape, restoreFocus = true, skipAutoFocus = false, ...props }) => {
    const trappedRef = useRef(null);
    /**
     * Restore the focus to the previously focused element (often, it's the CTA that opened the trap)
     */
    useEffect(() => {
        let currentFocus = null;
        if (restoreFocus) {
            currentFocus = document.activeElement;
        }
        return () => {
            if (currentFocus) {
                currentFocus.focus();
            }
        };
    }, [restoreFocus]);
    /**
     * Sends the focus to the first element of the focus trap tree
     */
    useEffect(() => {
        if (!trappedRef.current || skipAutoFocus)
            return;
        const focusableChildren = getFocusableNodes(trappedRef.current);
        if (focusableChildren.length > 0) {
            const firstElement = focusableChildren[0];
            firstElement.focus();
        }
        else {
            console.warn('[FocusTrap]: it seems there are no focusable elements in the focus trap tree. Make sure there s at least one.');
        }
    }, [skipAutoFocus]);
    const handleKeyDown = (e) => {
        if (e.key === KeyboardKeys.ESCAPE && onEscape) {
            onEscape();
            return;
        }
        if (e.key !== KeyboardKeys.TAB)
            return;
        const focusableChildren = getFocusableNodes(trappedRef.current);
        if (focusableChildren.length > 0) {
            const firstElement = focusableChildren[0];
            const lastElement = focusableChildren[focusableChildren.length - 1];
            // e.shiftKey allows to verify reverse tab
            if (e.shiftKey) {
                if (firstElement === document.activeElement) {
                    e.preventDefault();
                    lastElement.focus();
                }
            }
            else if (lastElement === document.activeElement) {
                e.preventDefault();
                firstElement.focus();
            }
        }
    };
    // eslint-disable-next-line jsx-a11y/no-static-element-interactions
    return jsx("div", { ref: trappedRef, onKeyDown: handleKeyDown, ...props });
};

export { FocusTrap };
