import { jsxs, jsx } from 'react/jsx-runtime';
import { forwardRef } from 'react';
import styled, { css } from 'styled-components';
import { useField } from './FieldContext.mjs';
import { inputFocusStyle } from '../themes/utils.mjs';
import { Box } from '../Box/Box.mjs';
import { Flex } from '../Flex/Flex.mjs';

// padding-[top|bottom] must ensure, the input matches the height of getThemeSize('input')
const PADDING_Y = {
    S: 6.5,
    M: 10.5,
};
const FieldInput = forwardRef(({ endAction, startAction, disabled = false, onChange, size = 'M', ...props }, ref) => {
    const { id, error, hint, name, required } = useField();
    let ariaDescription;
    if (error) {
        ariaDescription = `${id}-error`;
    }
    else if (hint) {
        ariaDescription = `${id}-hint`;
    }
    const hasError = Boolean(error);
    const handleChange = (e) => {
        if (!disabled && onChange) {
            onChange(e);
        }
    };
    return (jsxs(InputWrapper, { justifyContent: "space-between", hasError: hasError, disabled: disabled, children: [startAction ? (jsx(Box, { paddingLeft: 3, paddingRight: 2, children: startAction })) : null, jsx(Input, { id: id, name: name, ref: ref, "aria-describedby": ariaDescription, "aria-invalid": hasError, "aria-disabled": disabled, disabled: disabled, "data-disabled": disabled ? '' : undefined, hasLeftAction: Boolean(startAction), hasRightAction: Boolean(endAction), onChange: handleChange, "aria-required": required, "$size": size, ...props }), endAction ? (jsx(Box, { paddingLeft: 2, paddingRight: 3, children: endAction })) : null] }));
});
const Input = styled.input `
  border: none;
  border-radius: ${({ theme }) => theme.borderRadius};
  padding-bottom: ${({ $size }) => `${PADDING_Y[$size] / 16}rem`};
  padding-left: ${({ theme, hasLeftAction }) => (hasLeftAction ? 0 : theme.spaces[4])};
  padding-right: ${({ theme, hasRightAction }) => (hasRightAction ? 0 : theme.spaces[4])};
  padding-top: ${({ $size }) => `${PADDING_Y[$size] / 16}rem`};
  cursor: ${(props) => (props['aria-disabled'] ? 'not-allowed' : undefined)};

  color: ${({ theme }) => theme.colors.neutral800};
  font-weight: 400;
  font-size: ${(props) => props.theme.fontSizes[2]};
  display: block;
  width: 100%;
  background: inherit;

  ::placeholder {
    color: ${({ theme }) => theme.colors.neutral500};
    opacity: 1;
  }

  &[aria-disabled='true'] {
    color: inherit;
  }

  //focus managed by InputWrapper
  &:focus {
    outline: none;
    box-shadow: none;
  }
`;
const InputWrapper = styled(Flex) `
  border: 1px solid ${({ theme, hasError }) => (hasError ? theme.colors.danger600 : theme.colors.neutral200)};
  border-radius: ${({ theme }) => theme.borderRadius};
  background: ${({ theme }) => theme.colors.neutral0};
  ${inputFocusStyle()}

  ${({ theme, disabled }) => disabled
    ? css `
          color: ${theme.colors.neutral600};
          background: ${theme.colors.neutral150};
        `
    : undefined}
`;

export { FieldInput, InputWrapper };
