import { ExtendedType, Operation, Path, Point, PointEntry } from '..';
import { RangeDirection } from '../types/types';
/**
 * `Range` objects are a set of points that refer to a specific span of a Slate
 * document. They can define a span inside a single node or a can span across
 * multiple nodes.
 */
export interface BaseRange {
    anchor: Point;
    focus: Point;
}
export declare type Range = ExtendedType<'Range', BaseRange>;
export interface RangeEdgesOptions {
    reverse?: boolean;
}
export interface RangeTransformOptions {
    affinity?: RangeDirection | null;
}
export interface RangeInterface {
    /**
     * Get the start and end points of a range, in the order in which they appear
     * in the document.
     */
    edges: (range: Range, options?: RangeEdgesOptions) => [Point, Point];
    /**
     * Get the end point of a range.
     */
    end: (range: Range) => Point;
    /**
     * Check if a range is exactly equal to another.
     */
    equals: (range: Range, another: Range) => boolean;
    /**
     * Check if a range includes a path, a point or part of another range.
     */
    includes: (range: Range, target: Path | Point | Range) => boolean;
    /**
     * Get the intersection of a range with another.
     */
    intersection: (range: Range, another: Range) => Range | null;
    /**
     * Check if a range is backward, meaning that its anchor point appears in the
     * document _after_ its focus point.
     */
    isBackward: (range: Range) => boolean;
    /**
     * Check if a range is collapsed, meaning that both its anchor and focus
     * points refer to the exact same position in the document.
     */
    isCollapsed: (range: Range) => boolean;
    /**
     * Check if a range is expanded.
     *
     * This is the opposite of [[Range.isCollapsed]] and is provided for legibility.
     */
    isExpanded: (range: Range) => boolean;
    /**
     * Check if a range is forward.
     *
     * This is the opposite of [[Range.isBackward]] and is provided for legibility.
     */
    isForward: (range: Range) => boolean;
    /**
     * Check if a value implements the [[Range]] interface.
     */
    isRange: (value: any) => value is Range;
    /**
     * Iterate through all of the point entries in a range.
     */
    points: (range: Range) => Generator<PointEntry, void, undefined>;
    /**
     * Get the start point of a range.
     */
    start: (range: Range) => Point;
    /**
     * Transform a range by an operation.
     */
    transform: (range: Range, op: Operation, options?: RangeTransformOptions) => Range | null;
}
export declare const Range: RangeInterface;
//# sourceMappingURL=range.d.ts.map