'use strict';

Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });

const jsxRuntime = require('react/jsx-runtime');
const React = require('react');
const langJson = require('@codemirror/lang-json');
const reactCodemirror = require('@uiw/react-codemirror');
const styled = require('styled-components');
const decorationExtension = require('./utils/decorationExtension.js');
const useComposeRefs = require('../hooks/useComposeRefs.js');
const Field = require('../Field/Field.js');
const Flex = require('../Flex/Flex.js');
const FieldLabel = require('../Field/FieldLabel.js');
const FieldError = require('../Field/FieldError.js');
const FieldHint = require('../Field/FieldHint.js');
const utils = require('../themes/utils.js');

const _interopDefault = e => e && e.__esModule ? e : { default: e };

const styled__default = /*#__PURE__*/_interopDefault(styled);

const JSONInput = React.forwardRef(({ label, error, hint, labelAction, value = '', required = false, disabled = false, onChange = () => null, ...boxProps }, forwardedRef) => {
    const editor = React.useRef();
    const editorState = React.useRef();
    const editorView = React.useRef();
    const hasError = Boolean(error);
    /**
     * Determines the line to highlight when lintJSON finds an error via jsonParseLinter()
     */
    const highglightErrorAtLine = (lineNumber) => {
        const doc = editorState.current?.doc;
        if (doc) {
            const { text, to: lineEnd } = doc.line(lineNumber);
            const lineStart = lineEnd - text.trimStart().length;
            if (lineEnd > lineStart) {
                editorView.current?.dispatch({
                    effects: decorationExtension.addMarks.of([decorationExtension.lineHighlightMark.range(lineStart, lineEnd)]),
                });
            }
        }
    };
    const clearErrorHighlight = () => {
        const doc = editorState.current?.doc;
        if (doc) {
            const docEnd = doc.length || 0;
            editorView.current?.dispatch({
                effects: decorationExtension.filterMarks.of((from, to) => to <= 0 || from >= docEnd),
            });
        }
    };
    /**
     * Checks code editor for valid json input and then highlights any errors
     */
    const lintJSON = ({ state, view }) => {
        editorView.current = view;
        editorState.current = state;
        clearErrorHighlight();
        // Function calls json.parse and returns error message + position
        const lintJSONForErrrors = langJson.jsonParseLinter();
        const lintErrors = lintJSONForErrrors(view);
        if (lintErrors.length) {
            highglightErrorAtLine(state.doc.lineAt(lintErrors[0].from).number);
        }
    };
    const onCodeMirrorChange = (currentValue, viewUpdate) => {
        lintJSON(viewUpdate);
        // Call the parent's onChange handler
        onChange(currentValue);
    };
    const onCreateEditor = (view, state) => {
        editorView.current = view;
        editorState.current = state;
        // Lint the JSON in case the initial value is invalid
        lintJSON({ view, state });
    };
    const { setContainer, view } = reactCodemirror.useCodeMirror({
        value,
        onCreateEditor,
        container: editor.current,
        editable: !disabled,
        extensions: [langJson.json(), decorationExtension.markField],
        onChange: onCodeMirrorChange,
        theme: 'dark',
        basicSetup: {
            lineNumbers: true,
            bracketMatching: true,
            closeBrackets: true,
            indentOnInput: true,
            syntaxHighlighting: true,
            highlightSelectionMatches: true,
            tabSize: 2,
        },
    });
    const focusInput = () => {
        if (!disabled && view) {
            view.focus();
        }
    };
    const composedRefs = useComposeRefs.useComposedRefs(editor, setContainer);
    React.useImperativeHandle(forwardedRef, () => ({
        ...view?.dom,
        focus() {
            if (view) {
                view.focus();
            }
        },
        scrollIntoView(args) {
            if (view) {
                view.dom.scrollIntoView(args);
            }
        },
    }), [view]);
    return (jsxRuntime.jsx(Field.Field, { error: error, hint: hint, required: required, children: jsxRuntime.jsxs(Flex.Flex, { direction: "column", alignItems: "stretch", gap: 1, children: [label && (jsxRuntime.jsx(FieldLabel.FieldLabel, { onClick: focusInput, action: labelAction, children: label })), jsxRuntime.jsx(JSONInputContainer, { ref: composedRefs, hasError: hasError, alignItems: "stretch", fontSize: 2, hasRadius: true, ...boxProps }), jsxRuntime.jsx(FieldError.FieldError, {}), jsxRuntime.jsx(FieldHint.FieldHint, {})] }) }));
});
const JSONInputContainer = styled__default.default(Flex.Flex) `
  line-height: ${({ theme }) => theme.lineHeights[2]};

  .cm-editor {
    /** 
     * Hard coded since the color is the same between themes,
     * theme.colors.neutral800 changes between themes 
     */
    background-color: #32324d;
    width: 100%;
    outline: none;
  }

  .cm-scroller {
    border: 1px solid ${({ theme, hasError }) => (hasError ? theme.colors.danger600 : theme.colors.neutral200)};
    /* inputFocusStyle will receive hasError prop */
    ${utils.inputFocusStyle()}
  }

  .cm-editor,
  .cm-scroller {
    border-radius: ${({ theme }) => theme.borderRadius};
  }

  .cm-gutters,
  .cm-activeLineGutter {
    /** 
     * Hard coded since the color is the same between themes,
     * theme.colors.neutral700 changes between themes 
     */
    background-color: #4a4a6a;
  }
`;

exports.JSONInput = JSONInput;
